// ProcesadorBuffers.js
// Este archivo se encarga de procesar los buffers inactivos, si cumplen condiciones de tiempo y exclusividad
const redis = require('../config/redis');
const tokenizer = require('./MaruTokenizer');
const IAController = require('../services/IAController');

const TTL_INACTIVIDAD = 15; // Tiempo mínimo de inactividad para procesar un buffer
const MAX_CONCURRENTES = 10;

async function procesarBuffersInactivos() {
  const { default: pLimit } = await import('p-limit');
  const limit = pLimit(MAX_CONCURRENTES);
  const keys = await redis.keys('buffer:*');

  await Promise.all(
    keys.map(key => limit(async () => {
      try {
        const raw = await redis.get(key);
        if (!raw) return;

        const data = JSON.parse(raw);
        const { mensajes, procesando } = data;

        // Si ya está marcado como procesando, lo ignoramos
        if (procesando) {
          console.log(`[${key}] Ya está marcado como procesando, ignorado.`);
          return;
        }

        const ultimo = mensajes[mensajes.length - 1]?.timestamp || 0;
        const inactivoDesde = (Date.now() - ultimo) / 1000;

        const telefono = key.split(':')[1];
        console.log(`[${telefono}] Inactivo desde: ${inactivoDesde.toFixed(2)} segundos`);

        if (inactivoDesde >= TTL_INACTIVIDAD) {
          const lockKey = `lock:${key}`;

          // Intentamos obtener el lock (60s para asegurar procesamiento único)
          const gotLock = await redis.set(lockKey, '1', { NX: true, EX: 60 });
          if (!gotLock) {
            console.log(`[${telefono}] Ya está siendo procesado por otro ciclo, ignorado.`);
            return;
          }

          // Marcamos el buffer como procesando y le damos un TTL de 5 minutos
          await redis.set(key, JSON.stringify({ ...data, procesando: true }), 'EX', 300);

          const textoFinal = mensajes.map(m => m.mensaje).join(' ');
          console.log(`[${telefono}] Procesando con IA: "${textoFinal}"`);

          await IAController.responder(telefono, textoFinal);

          // Eliminamos el buffer después del procesamiento
          await redis.del(key);
          console.log(`[${telefono}] Respondido y buffer eliminado`);
        }
      } catch (err) {
        console.error(`Error procesando ${key}:`, err.message);
      }
    }))
  );
}

module.exports = { procesarBuffersInactivos };

