const WhatsAppService = require('./whatsapp');
const Conversacion = require('../models/Conversacion');
const tokenizer = require('./MaruTokenizer');

const NegocioController = require('../controllers/NegocioController');
const NegocioClass = require('../clases/Negocio');
const SessionController = require('../controllers/SessionController');
const { responderConPrioridad } = require('../services/IASelectorService');

const IAController = {
  async responder(telefono, textoFinal) {
    console.log(`[${telefono}] Iniciando respuesta IA con texto: "${textoFinal}"`);

    let respuesta = '';
    let fuente = 'Desconocida';
    let negocio = null;
    let nombreNegocio = 'Maru';
    let valorPrioridad = 2; // Default: 'pro'

    try {
      console.log(`[${telefono}] Consultando sesión activa...`);
      const sesion = await SessionController.obtenerSesionActivaPorTelefono(telefono);

      if (sesion) {
        console.log(`[${telefono}] Sesión encontrada: ${JSON.stringify(sesion)}`);

        console.log(`[${telefono}] Buscando información del negocio con esquema: ${sesion.esquema_negocio}`);
        const data = await NegocioController.obtenerNegocioPorTelefonoInterno(sesion.esquema_negocio);

        if (!data) {
          console.warn(`[${telefono}] No se encontró información del negocio.`);
          respuesta = 'No se encontró información del negocio asignado.';
          fuente = 'SISTEMA:';
        } else {
          negocio = new NegocioClass(data);
          nombreNegocio = negocio.nombre_negocio;
          console.log(`[${telefono}] Negocio instanciado: ${nombreNegocio}`);

          if (!negocio.esActivo()) {
            console.warn(`[${telefono}] El negocio está inactivo.`);
            respuesta = 'El negocio con el que estás hablando está inactivo.';
            fuente = 'Sistema';
          } else {
            const plan = negocio.tipo_plan?.toLowerCase();
            console.log(`[${telefono}] Plan del negocio: ${plan}`);

            if (plan === 'starter') valorPrioridad = 1;
            else if (plan === 'pro') valorPrioridad = 2;
            else if (plan === 'enterprise') valorPrioridad = 3;

            console.log(`[${telefono}] Prioridad IA: ${valorPrioridad}`);

            try {
              const resultado = await Promise.race([
                responderConPrioridad(valorPrioridad, textoFinal, negocio),
                new Promise((_, reject) => setTimeout(() => reject(new Error('Timeout IA')), 15000))
              ]);
              respuesta = resultado.respuesta;
              fuente = resultado.fuente;
              console.log(`[${telefono}] IA respondió desde ${fuente}`);
            } catch (timeoutError) {
              console.warn(`[${telefono}] Timeout alcanzado. Respuesta genérica enviada.`);
              respuesta = 'No puedo responder en este momento';
              fuente = 'Sistema';
            }
          }
        }

      } else {
        console.log(`[${telefono}] No hay sesión activa. Usando IA por defecto (prioridad 2)`);

        try {
          const resultado = await Promise.race([
            responderConPrioridad(2, textoFinal),
            new Promise((_, reject) => setTimeout(() => reject(new Error('Timeout IA')), 15000))
          ]);
          respuesta = resultado.respuesta;
          fuente = resultado.fuente;
          console.log(`[${telefono}] IA respondió desde ${fuente} (modo orquestadora)`);
        } catch (timeoutError) {
          console.warn(`[${telefono}] Timeout alcanzado. Respuesta genérica enviada.`);
          respuesta = 'No puedo responder en este momento';
          fuente = 'Sistema';
        }
      }

    } catch (error) {
      console.error(`[${telefono}] Error en IAController:`, error.message);
      respuesta = 'No puedo responder en este momento';
      fuente = 'Sistema';
    }

    console.log(`[${telefono}] Guardando conversación...`);
    await Conversacion.guardar(telefono, 'enviado', respuesta, nombreNegocio, Date.now());

    const { tokens } = tokenizer.medir(respuesta);
    console.log(`[${telefono}] Total tokens: ${tokens}`);

    console.log(`[${telefono}] Enviando mensaje por WhatsApp...`);
    await WhatsAppService.responderTexto(
      telefono,
      `Maru de *${nombreNegocio}* ${tokens} ${fuente}\n\n${respuesta}`
    );

    console.log(`[${telefono}] Respuesta completada y enviada.`);
  }
};

module.exports = IAController;

